function rimless_wheel_grf
%% Tabula Rasa
% clear all
% close all

%% Parameters
g = 9.8;		%gravity [m/s]
l = 1;		%leg length [m]
m = 1;		%point mass [kg]
alpha = pi/8;		%half the angle between spokes [rad]
pitch = .15;% 0.15;		%pitch of ramp [rad]

%% Initial Conditions
theta_dot0 = 4;		%start velocity [rad/s]
number_steps = 10;		%number of steps

omega1 = sqrt(2*g*(1 - cos(pitch - alpha)/l));	%minimum speed to vault mass over stance leg [rad/s]

if theta_dot0 >= 0		%setting theta for appropriate pivot leg
	theta0 = pitch - alpha;
else
	theta0 = pitch + alpha;
end

%% Switching event
zero_crossing = odeset('Events',@leg_switch);	%function to detect leg switch

theta = theta0;
theta_dot = theta_dot0;

%% Phase plot figure
figure(11)
clf
% fighan = gcf;
% axishan = gca;
title('Phase Portrait Trajectories of a Rimless Wheel')
xlabel('\theta [rad]')
ylabel('d\theta/dt [rad/s]')
grid('on')
box('off')
hold on

axis([-0.3 0.6 -0.5 5])

switchhan(1) = line([pitch+alpha pitch+alpha],[-1 4.5]);
switchhan(2) = line([pitch-alpha pitch-alpha],[-1 4.5]);
set(switchhan,'color',[0 0 0],'linewidth',1,'linestyle','--');

starthan = plot(theta0,theta_dot0);		%red cross marks start
set(starthan,'marker','x','markersize',10,'linewidth',1.5,'color',[1 0 0]);

modehan(1) = line([(pitch-alpha) (alpha+pitch)],[-omega1 omega1*(alpha+pitch)/(alpha-pitch)]);
modehan(2) = line([(pitch-alpha) (alpha+pitch)],[omega1 -omega1*(alpha+pitch)/(alpha-pitch)]);
set(modehan,'color',[0.7 0.7 0.7])


%% Phase plot simulation

for i = 1:number_steps
	[Step(i).t,Step(i).q] = ode45(@EoM, [0 2], [theta theta_dot],zero_crossing);
	theta_dot = Step(i).q(end,2)*cos(2*alpha);
	
	if theta_dot >= 0
		theta = pitch - alpha;
	else
		theta = pitch + alpha;
	end	
	
	stephan = phase_plot(Step(i),theta,theta_dot);
end

limithan = limit_cycle(Step(i),theta,theta_dot);
h = [stephan, limithan];
legend(h,'Phase plot','Limit Cycle')

%% Output Figure - uncomment to print a .eps file.
% print rimless_wheel_phase_plot.eps -depsc2


%% Return Map

figure(12)
clf
title('Return Map of a Rimless Wheel')
xlabel('d\theta^+/dt [rad/s]')
ylabel('d\theta^-/dt [rad/s]')
grid('on')
box('off')
hold on
axis([0 theta_dot0 0 theta_dot0])

theta_dot_n = 0:0.1:theta_dot0;
Return_Map = @(theta) cos(2*alpha)*sqrt(theta.^2 + 4*g*sin(alpha)*sin(pitch)/l);
returnhan = plot(theta_dot_n, Return_Map(theta_dot_n),'r','linewidth',1.5);
line([0 theta_dot0], [0 theta_dot0],'linewidth',1.5)

for i = 1:number_steps-2
    simhan = line([Step(i).q(1,2) Step(i+1).q(1,2)],[Step(i+1).q(1,2) Step(i+1).q(1,2)],'color','g','linewidth',1.5);
    line([Step(i+1).q(1,2) Step(i+1).q(1,2)],[Step(i+1).q(1,2) Step(i+2).q(1,2)],'color','g','linewidth',1.5)
end

omega_roll = cot(2*alpha)*sqrt(4*g*sin(alpha)*sin(pitch));
% plot(omega_roll,omega_roll,'rx','markersize',10,'linewidth',2)
annotation('textarrow',[1.2/theta_dot0 1.11*omega_roll/theta_dot0],[2/theta_dot0 1.11*omega_roll/theta_dot0],'String','Fixed Point','FontSize',16)
h = [returnhan, simhan];
legend(h,'Return Map','Simulation Results','location','northwest')
print figures/return_map.eps -depsc2

%% Ground reaction force

% Calculate accelerations for last step:
for t = 1:length(Step(end).t)
    q_dot_temp = EoM(1,Step(end).q(t,:));
    q_dot(t) = q_dot_temp(1);
    q_ddot(t) = q_dot_temp(2);
    q(t) = Step(end).q(t,1);
end
% Fx = m*q_ddot*l*cos(q(t));
% Fy = -m*q_ddot*l*sin(q(t)) - m*q_dot*l*+m*g;

x_ddot = l*(+q_ddot.*cos(q) - q_dot.^2.*sin(q));
y_ddot = l*(-q_ddot.*sin(q) - q_dot.^2.*cos(q));

% figure(14)
% clf
% hold on
% t = Step(end).t;
% plot(t,x_ddot,'k');
% plot(t,y_ddot,'r');
% leg_h = legend('$$\ddot{x}$$','$$\ddot{y}$$');
% set(leg_h,'interpreter', 'latex');

Fx = m*x_ddot;
Fy = m*y_ddot + m*g;

Fx = Fx/(m*g);
Fy = Fy/(m*g);

% Rotate by slope angle ? to get horizontal and vertical components
M = [cos(pitch) -sin(pitch);sin(pitch) cos(pitch)];
Fhv = M*[Fx;Fy];

% Plot
h_grf_xy = figure(13);
clf
hold on
t = Step(end).t/Step(end).t(end);
plot(t,Fx,'k');
plot(t,Fy,'r');
xlabel('relative step time [-]')
ylabel('relative ground reaction force [-]')
legend('F_x','F_y')

h_grf_hv = figure(14);
clf
hold on
plot(t,Fhv(1,:),'k');
plot(t,Fhv(2,:),'r');
xlabel('relative step time [-]')
ylabel('relative ground reaction force [-]')
legend('F_H','F_V')
%% Equation of Motion
function output = EoM(~,q)
	
	output(1,1) = q(2);
	output(2,1) = g*sin(q(1))/l;
end

%% Switch condition
function [impact,terminate,direction] = leg_switch(~,q)

	impact = abs(q(1) - pitch) - alpha;
	terminate = 1;
	direction = 1;
end

%% Phase plot
function stephan = phase_plot(Step,theta,theta_dot)
	linehan = line([Step.q(end,1) theta],[Step.q(end,2) theta_dot]);
	set(linehan,'color',[0.7 0.7 0.7])
	stephan = plot(Step.q(:,1),Step.q(:,2));
	set(stephan,'color',[0 0 0],'linewidth',1.5)
end

%% Limit cycle
function stephan = limit_cycle(Step,theta,theta_dot)
	linehan = line([Step.q(end,1) theta],[Step.q(end,2) theta_dot]);
	set(linehan,'color',[1 0 0],'linewidth',2)
	stephan = plot(Step.q(:,1),Step.q(:,2));
	set(stephan,'color',[1 0 0],'linewidth',2)
end
end